<?xml version="1.0"?>
<!--
* Copyright 2023 IBM Corporation
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
-->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:dp="http://www.datapower.com/extensions"
    xmlns:apim="http://www.ibm.com/apimanagement"
    xmlns:json="http://www.ibm.com/xmlns/prod/2009/jsonx"
    version="1.0" extension-element-prefixes="dp" exclude-result-prefixes="dp json apim">
    
  <!-- Contains the APIM functions --> 
  <xsl:include href="store:///dp/apim.custom.xsl" />
  <!-- If running in a v5 compatible gateway, the xsl:include should instead be:
  <xsl:include href="local:///isp/policy/apim.custom.xsl" />
  
  If only one gateway type needs to be supported, change the include appropriately.
  
  If you require one rule that can be called from either gateway type, the implication is
  a different stylesheet, and thus a different AAA action, will need to be used when
  the rule is executed from a v5c gateway (MPGW service based) versus an API Gateway
  (API Gateway service based) because of the xsl:include difference, either of which
  would produce the dynamic AAA action. The rule would require an initial stylesheet to
  simply output var://service/processor-type as part of a simple XML output to a named
  context, for example,
  
  <type>Multiprotocol Gateway</type>
  
  and then use a conditional action checking that XML in the same named context with an xpath
  /type = 'Multiprotocol Gateway' to drive the v5c AAA action
  /type != 'Multiprotocol Gateway' to drive the APIGW AAA action.
  -->

  <xsl:variable name="log-category" select="'apiconnect'" />

  <xsl:template match="/">
    <!-- Retrieves node-set containing all Policy properties -->
    <xsl:variable name="properties" select="dp:stringToJSONx(dp:variable('var://context/msRuleContext/properties'))" />

    <!-- verbose is also provided by the rule bridge code to determine if non error messages should be logged -->
    <xsl:variable name="verbose" select="dp:variable('var://context/msRuleContext/verbose')" />

    <!-- if the truststore property is provided in the UDP properties, then resolve the APIM tlsClientProfile name
         to the DP tlsClientProfile name. -->
    <xsl:variable name="tlsClientProfile">
      <xsl:variable name="truststore" select="$properties/json:object/json:string[@name = 'truststore']" />
      <xsl:if test="$verbose = 'true'">
        <xsl:message dp:priority="debug" dp:type="{$log-category}"><xsl:value-of select="dp:variable('var://service/transaction-rule-name')"/> truststore=[<xsl:value-of select="$truststore"/>]</xsl:message>
      </xsl:if>
      <xsl:if test="string-length($truststore) &gt; 0">
        <!-- Note that by convention in the API Gateway, the valcred object has the same name as the
             tls client profile object except for the -valcred suffix. In a v5 compatible Gateway, the
             valcred object has the same name as the tls client profile object without a -valcred suffix and
             the concat of -valcred should not be done. -->
        <xsl:variable name="tlsProfileObjName" select="concat(apim:getTLSProfileObjName($truststore),'-valcred')" />
        <xsl:choose>
          <!-- The getTLSProfileObjName function should append org/catalog obfuscated names with client: which must not be provided 
               for the AAA val cred name for the name of the validation credential object -->
          <xsl:when test="starts-with($tlsProfileObjName, 'client:')">
            <xsl:value-of select="substring-after($tlsProfileObjName, 'client:')" />
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="$tlsProfileObjName" />
          </xsl:otherwise>
        </xsl:choose>
      </xsl:if>
    </xsl:variable>

    <xsl:if test="$verbose = 'true'">
      <xsl:message dp:priority="debug" dp:type="{$log-category}"><xsl:value-of select="dp:variable('var://service/transaction-rule-name')"/> tlsClientProfile =[<xsl:value-of select="$tlsClientProfile" />]</xsl:message>
    </xsl:if>

    <!-- update this AAA policy template by dynamically adding the validation credential name (which by 
         convention has the same name as the tls client profile except For the -valcred suffix -->
    <xsl:variable name="templateAAAPolicy" select="/input/template-aaa-config/AAAPolicy" />
    <result>
      <AAAPolicy name="{$templateAAAPolicy/@name}">
        <SAMLValcred class="CryptoValCred"><xsl:value-of select="$tlsClientProfile"/></SAMLValcred>
      </AAAPolicy>
      <merge-config>on</merge-config>
      <validate-config>off</validate-config>
    </result>
  </xsl:template> 
</xsl:stylesheet>